<?php
require 'db.php';

/* ----------------------------------------------------------
   0) Helper de upload (só define se não existir)
---------------------------------------------------------- */
if (!function_exists('fazer_upload_arquivo')) {
    function fazer_upload_arquivo($file) {
        $dir = __DIR__ . '/uploads/comprovantes';
        if (!is_dir($dir)) {
            @mkdir($dir, 0777, true);
        }
        $nomeOriginal = preg_replace('/[^a-zA-Z0-9._-]/', '_', $file['name']);
        $ext = pathinfo($nomeOriginal, PATHINFO_EXTENSION);
        $alvo = uniqid('comp_', true) . ($ext ? '.' . strtolower($ext) : '');
        $destino = $dir . '/' . $alvo;
        if (move_uploaded_file($file['tmp_name'], $destino)) {
            return $alvo; // salva só o nome do arquivo
        }
        return null;
    }
}

/* ----------------------------------------------------------
   1) FILTRO DE DATA (ANO/MÊS)
---------------------------------------------------------- */
$ano_selecionado = isset($_GET['ano']) ? (int)$_GET['ano'] : date('Y');
$mes_selecionado = isset($_GET['mes']) ? (int)$_GET['mes'] : (int)date('m');

$params = [];
$tipos_params = '';
if ($mes_selecionado != 0) {
    $filtro_sql = "WHERE YEAR(data_transacao) = ? AND MONTH(data_transacao) = ?";
    $params = [$ano_selecionado, $mes_selecionado];
    $tipos_params = 'ii';
} else {
    $filtro_sql = "WHERE YEAR(data_transacao) = ?";
    $params = [$ano_selecionado];
    $tipos_params = 'i';
}

/* ----------------------------------------------------------
   2) AÇÕES (CRIAR / EXCLUIR / LIMPAR)
---------------------------------------------------------- */
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST['acao'])) {
        switch ($_POST['acao']) {
            case 'adicionar_transacao':
                $tipo_e_origem = explode(':', $_POST['tipo_origem']);
                $tipo = $tipo_e_origem[0] ?? '';
                $origem = $tipo_e_origem[1] ?? '';
                $data_transacao = $_POST['data_transacao'] ?? '';
                $descricao = trim($_POST['descricao'] ?? '');
                $valor_str = str_replace(',', '.', $_POST['valor'] ?? '0');
                $valor = (float)$valor_str;
                $forma_pagamento = !empty($_POST['forma_pagamento']) ? trim($_POST['forma_pagamento']) : null;
                $entidade = !empty($_POST['entidade']) ? trim($_POST['entidade']) : null;

                // comissão (mantido para compatibilidade)
                $comissao_para = null;
                if (!empty($_POST['comissao_para'])) {
                    $comissao_para = trim($_POST['comissao_para']);
                } elseif (!empty($_POST['comissao_para_custom'])) {
                    $comissao_para = trim($_POST['comissao_para_custom']);
                }
                if ($origem === 'COMISSOES' && !empty($comissao_para)) {
                    $entidade = $comissao_para;
                }

                // upload do comprovante
                $comprovante_nome = null;
                if (isset($_FILES['comprovante_pagamento']) && $_FILES['comprovante_pagamento']['error'] == UPLOAD_ERR_OK) {
                    $comprovante_nome = fazer_upload_arquivo($_FILES['comprovante_pagamento']);
                }

                if (!empty($data_transacao) && !empty($tipo) && !empty($origem) && !empty($descricao) && $valor > 0) {
                    $sql_insert = "INSERT INTO transacoes (data_transacao, tipo, descricao, valor, forma_pagamento, comprovante_pagamento, entidade, origem)
                                   VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
                    $stmt_insert = $conexao->prepare($sql_insert);
                    $stmt_insert->bind_param('sssdssss', $data_transacao, $tipo, $descricao, $valor, $forma_pagamento, $comprovante_nome, $entidade, $origem);
                    if ($stmt_insert->execute()) {
                        header("Location: index.php?status=success&ano=$ano_selecionado&mes=$mes_selecionado");
                        exit();
                    } else {
                        header("Location: index.php?status=error&ano=$ano_selecionado&mes=$mes_selecionado");
                        exit();
                    }
                } else {
                    header("Location: index.php?status=validation_error&ano=$ano_selecionado&mes=$mes_selecionado");
                    exit();
                }
                break;

            case 'excluir_transacao':
                $id = (int)($_POST['id'] ?? 0);
                if ($id > 0) {
                    // apaga arquivo de comprovante se existir
                    $stmt_buscar = $conexao->prepare("SELECT comprovante_pagamento FROM transacoes WHERE id = ?");
                    $stmt_buscar->bind_param('i', $id);
                    $stmt_buscar->execute();
                    $resultado = $stmt_buscar->get_result();
                    if ($transacao = $resultado->fetch_assoc()) {
                        $nomeArq = $transacao['comprovante_pagamento'] ?? null;
                        if ($nomeArq && file_exists(__DIR__ . '/uploads/comprovantes/' . $nomeArq)) {
                            @unlink(__DIR__ . '/uploads/comprovantes/' . $nomeArq);
                        }
                    }

                    $stmt_delete = $conexao->prepare("DELETE FROM transacoes WHERE id = ?");
                    $stmt_delete->bind_param('i', $id);
                    if ($stmt_delete->execute()) {
                        header("Location: index.php?status=deleted&ano=$ano_selecionado&mes=$mes_selecionado");
                        exit();
                    }
                }
                header("Location: index.php?status=error&ano=$ano_selecionado&mes=$mes_selecionado");
                exit();

            case 'limpar_todas_transacoes':
                if (isset($_POST['confirmar_limpeza']) && $_POST['confirmar_limpeza'] === 'SIM') {
                    // remove todos os comprovantes
                    $comprovantes = $conexao->query("SELECT comprovante_pagamento FROM transacoes WHERE comprovante_pagamento IS NOT NULL");
                    while ($comp = $comprovantes->fetch_assoc()) {
                        $nomeArq = $comp['comprovante_pagamento'];
                        if ($nomeArq && file_exists(__DIR__ . '/uploads/comprovantes/' . $nomeArq)) {
                            @unlink(__DIR__ . '/uploads/comprovantes/' . $nomeArq);
                        }
                    }
                    // limpa tabela
                    $conexao->query("TRUNCATE TABLE transacoes");
                    header("Location: index.php?status=all_deleted&ano=$ano_selecionado&mes=$mes_selecionado");
                    exit();
                }
                header("Location: index.php?status=error&ano=$ano_selecionado&mes=$mes_selecionado");
                exit();
        }
    }
}

/* ----------------------------------------------------------
   3) CONSULTAS PARA O DASHBOARD
---------------------------------------------------------- */
function executar_query_com_filtro($conexao, $sql, $params, $tipos_params) {
    $stmt = $conexao->prepare($sql);
    if (!empty($params)) {
        $stmt->bind_param($tipos_params, ...$params);
    }
    $stmt->execute();
    return $stmt->get_result();
}

$sql_receitas = "SELECT SUM(valor) as total FROM transacoes " . $filtro_sql . " AND tipo = 'Receita'";
$receitas_query = executar_query_com_filtro($conexao, $sql_receitas, $params, $tipos_params);
$total_receitas = $receitas_query->fetch_assoc()['total'] ?? 0;

$sql_despesas = "SELECT SUM(valor) as total FROM transacoes " . $filtro_sql . " AND tipo = 'Despesa'";
$despesas_query = executar_query_com_filtro($conexao, $sql_despesas, $params, $tipos_params);
$total_despesas = $despesas_query->fetch_assoc()['total'] ?? 0;

$saldo_total = (float)$total_receitas - (float)$total_despesas;

$sql_transacoes = "SELECT * FROM transacoes " . $filtro_sql . " ORDER BY data_transacao DESC, id DESC";
$transacoes = executar_query_com_filtro($conexao, $sql_transacoes, $params, $tipos_params);

$nomes_meses = [1=>'Janeiro',2=>'Fevereiro',3=>'Março',4=>'Abril',5=>'Maio',6=>'Junho',7=>'Julho',8=>'Agosto',9=>'Setembro',10=>'Outubro',11=>'Novembro',12=>'Dezembro'];
$titulo_relatorio = ($mes_selecionado == 0) ? "$ano_selecionado" : $nomes_meses[$mes_selecionado] . " de $ano_selecionado";
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>💰 Sistema Financeiro - Gabinete</title>

<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
<link rel="stylesheet" href="css/styles.css">
</head>
<body>

<div class="header">
  <div class="header-content">
    <div class="title-section">
      <h1><i class="fas fa-chart-line"></i>Sistema Financeiro</h1>
      <p>Gestão Completa de Receitas e Despesas</p>
    </div>
  </div>
</div>

<div class="container">
  <div class="navigation">
    <a href="index.php" class="nav-link nav-primary active">
      <i class="fas fa-plus-circle"></i>Lançar Contas
    </a>
    <a href="contas.php" class="nav-link nav-green">
      <i class="fas fa-list-check"></i>Controle de Contas
    </a>
    <a href="relatorios.php" class="nav-link nav-amber">
      <i class="fas fa-chart-bar"></i>Relatórios
    </a>
    <a href="http://localhost:8888/gabinete/" class="nav-link nav-primary nav-push-right" title="Voltar ao Sistema Principal" aria-label="Home">
      <i class="fas fa-home"></i>Sistema Principal
    </a>
  </div>

  <?php
    if (isset($_GET['status'])) {
        if ($_GET['status'] == 'success')        echo '<div class="alert alert-success"><i class="fas fa-check-circle"></i>Transação adicionada com sucesso!</div>';
        elseif ($_GET['status'] == 'deleted')     echo '<div class="alert alert-warning"><i class="fas fa-trash"></i>Transação excluída com sucesso!</div>';
        elseif ($_GET['status'] == 'all_deleted') echo '<div class="alert alert-warning"><i class="fas fa-broom"></i>Todas as transações foram excluídas!</div>';
        elseif ($_GET['status'] == 'error')       echo '<div class="alert alert-error"><i class="fas fa-exclamation-triangle"></i>Erro ao processar solicitação. Tente novamente.</div>';
        elseif ($_GET['status'] == 'validation_error') echo '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i>Preencha todos os campos obrigatórios.</div>';
    }
  ?>

  <div class="filter-container">
    <h2><i class="fas fa-calendar-alt"></i>Selecionar Período</h2>
    <form action="index.php" method="GET">
      <div class="form-row">
        <div class="form-group">
          <label for="mes"><i class="fas fa-calendar"></i>Mês</label>
          <select name="mes" id="mes">
            <option value="0" <?= ($mes_selecionado == 0) ? 'selected' : '' ?>>Todos os Meses (Anual)</option>
            <?php foreach ($nomes_meses as $num => $nome): ?>
              <option value="<?= $num ?>" <?= ($mes_selecionado == $num) ? 'selected' : '' ?>><?= $nome ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="form-group">
          <label for="ano"><i class="fas fa-calendar-year"></i>Ano</label>
          <select name="ano" id="ano">
            <?php for ($ano = date('Y'); $ano >= date('Y') - 5; $ano--): ?>
              <option value="<?= $ano ?>" <?= ($ano_selecionado == $ano) ? 'selected' : '' ?>><?= $ano ?></option>
            <?php endfor; ?>
          </select>
        </div>
        <div class="form-group">
          <button type="submit" class="btn-primary"><i class="fas fa-filter"></i>Filtrar</button>
        </div>
      </div>
    </form>
  </div>

  <div class="form-container">
    <h2><i class="fas fa-plus-circle"></i>Novo Lançamento</h2>
    <form action="index.php?ano=<?= $ano_selecionado ?>&mes=<?= $mes_selecionado ?>" method="post" enctype="multipart/form-data">
      <input type="hidden" name="acao" value="adicionar_transacao">
      <div class="form-row">
        <div class="form-group">
          <label for="data_transacao"><i class="fas fa-calendar-day"></i>Data*</label>
          <input type="date" id="data_transacao" name="data_transacao" value="<?= date('Y-m-d') ?>" required>
        </div>
        <div class="form-group">
          <label for="tipo_origem"><i class="fas fa-exchange-alt"></i>Tipo de Lançamento*</label>
          <select id="tipo_origem" name="tipo_origem" required>
            <option value="Despesa:DESPESAS">Despesas</option>
            <option value="Receita:RECEITA">Receita</option>
          </select>
        </div>
        <div class="form-group" style="grid-column: span 2;">
          <label for="descricao"><i class="fas fa-file-text"></i>Descrição*</label>
          <input type="text" id="descricao" name="descricao" required>
        </div>
        <div class="form-group">
          <label for="valor"><i class="fas fa-dollar-sign"></i>Valor (R$)*</label>
          <input type="text" id="valor" name="valor" placeholder="150,50" required>
        </div>
      </div>

      <div class="form-row">
        <div class="form-group" id="group-entidade" style="display:none;">
          <label for="entidade">Cliente / Fonte Pagadora</label>
          <input type="text" id="entidade" name="entidade">
        </div>
        <div class="form-group">
          <label for="forma_pagamento">Forma de Pagamento</label>
          <input type="text" id="forma_pagamento" name="forma_pagamento" placeholder="PIX, Cartão, Dinheiro, etc.">
        </div>
        <div class="form-group">
          <label for="comprovante_pagamento"><i class="fas fa-paperclip"></i>Comprovante (PDF/JPG/PNG)</label>
          <input type="file" id="comprovante_pagamento" name="comprovante_pagamento" accept=".jpg,.jpeg,.png,.pdf,.doc,.docx">
        </div>
      </div>

      <div class="form-row">
        <div class="form-group">
          <button type="submit" class="btn-primary"><i class="fas fa-save"></i>Adicionar Transação</button>
        </div>
      </div>
    </form>
  </div>

  <div class="report-content">
    <h2 style="text-align:center;"><?= htmlspecialchars($titulo_relatorio) ?></h2>
    <div class="summary">
      <div class="summary-box">
        <h3><i class="fas fa-arrow-up text-success"></i>Total de Receitas</h3>
        <p class="receita">R$ <?= number_format((float)$total_receitas, 2, ',', '.') ?></p>
      </div>
      <div class="summary-box">
        <h3><i class="fas fa-arrow-down text-danger"></i>Total de Despesas</h3>
        <p class="despesa">R$ <?= number_format((float)$total_despesas, 2, ',', '.') ?></p>
      </div>
      <div class="summary-box">
        <h3><i class="fas fa-balance-scale text-primary"></i>Saldo Final</h3>
        <p class="saldo">R$ <?= number_format((float)$saldo_total, 2, ',', '.') ?></p>
      </div>
    </div>

    <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:20px;">
      <div style="width:120px;"></div>
      <h3 style="margin:0;flex-grow:1;text-align:center;">Histórico de Transações do Período</h3>
      <button onclick="abrirModalLimparTudo()" class="btn-small btn-danger">🗑️ Limpar Todas</button>
    </div>

    <table>
      <thead>
        <tr>
          <th>Data</th><th>Origem</th><th>Descrição</th><th>Entidade</th><th>Forma Pgto</th><th>Comprovante</th>
          <th style="text-align:right;">Valor</th><th>Ações</th>
        </tr>
      </thead>
      <tbody>
        <?php while ($t = $transacoes->fetch_assoc()): ?>
          <tr>
            <td><?= date("d/m/Y", strtotime($t['data_transacao'])) ?></td>
            <td><?= htmlspecialchars($t['origem']) ?></td>
            <td><?= htmlspecialchars($t['descricao']) ?></td>
            <td><?= htmlspecialchars($t['entidade'] ?? '-') ?></td>
            <td><?= htmlspecialchars($t['forma_pagamento'] ?? '-') ?></td>
            <td>
              <?php if (!empty($t['comprovante_pagamento'])): ?>
                <a class="comprovante-link" href="uploads/comprovantes/<?= htmlspecialchars($t['comprovante_pagamento']) ?>" target="_blank">Ver</a>
              <?php else: ?> - <?php endif; ?>
            </td>
            <td class="valor <?= ($t['tipo'] === 'Receita') ? 'receita' : 'despesa' ?>">
              <?= ($t['tipo'] === 'Despesa' ? '-' : '+') ?> R$ <?= number_format((float)$t['valor'], 2, ',', '.') ?>
            </td>
            <td>
              <form method="post" style="display:inline" onsubmit="return confirm('Excluir esta transação?');">
                <input type="hidden" name="acao" value="excluir_transacao">
                <input type="hidden" name="id" value="<?= (int)$t['id'] ?>">
                <input type="hidden" name="ano" value="<?= (int)$ano_selecionado ?>">
                <input type="hidden" name="mes" value="<?= (int)$mes_selecionado ?>">
                <button class="btn-small btn-danger">🗑️</button>
              </form>
            </td>
          </tr>
        <?php endwhile; ?>
      </tbody>
    </table>
  </div>
</div>

<div id="modalExcluirTransacao" class="modal">
  <div class="modal-content">
    <h3>⚠️ Confirmar Exclusão</h3>
    <p>Tem certeza que deseja excluir esta transação?</p>
    <p><strong id="descricaoTransacao"></strong></p>
    <p style="color:#c53030;font-size:14px;">Esta ação não pode ser desfeita!</p>
    <form method="post" id="formExcluirTransacao">
      <input type="hidden" name="acao" value="excluir_transacao">
      <input type="hidden" name="id" id="idTransacao">
      <input type="hidden" name="ano" value="<?= (int)$ano_selecionado ?>">
      <input type="hidden" name="mes" value="<?= (int)$mes_selecionado ?>">
      <div class="modal-buttons">
        <button type="button" onclick="fecharModal('modalExcluirTransacao')" style="background:#6c757d;color:#fff;">Cancelar</button>
        <button type="submit" style="background:#dc3545;color:#fff;">Excluir</button>
      </div>
    </form>
  </div>
</div>

<div id="modalLimparTudo" class="modal">
  <div class="modal-content">
    <h3>🚨 ATENÇÃO - AÇÃO IRREVERSÍVEL</h3>
    <p><strong>Você está prestes a excluir TODAS as transações!</strong></p>
    <ul style="margin:15px 0;padding-left:20px;color:#c53030;">
      <li>Deletar todas as transações do banco</li>
      <li>Remover todos os comprovantes de pagamento</li>
      <li>Zerar todos os relatórios e estatísticas</li>
    </ul>
    <p style="color:#c53030;font-weight:bold;">ESTA AÇÃO NÃO PODE SER DESFEITA!</p>
    <form method="post" id="formLimparTudo">
      <input type="hidden" name="acao" value="limpar_todas_transacoes">
      <input type="hidden" name="ano" value="<?= (int)$ano_selecionado ?>">
      <input type="hidden" name="mes" value="<?= (int)$mes_selecionado ?>">
      <div style="margin:20px 0;padding:15px;background:#fff3cd;border-radius:8px;">
        <label style="display:flex;align-items:center;font-weight:bold;">
          <input type="checkbox" id="confirmarLimpeza" name="confirmar_limpeza" value="SIM" required style="margin-right:10px;">
          Eu confirmo que quero excluir TODAS as transações
        </label>
      </div>
      <div class="modal-buttons">
        <button type="button" onclick="fecharModal('modalLimparTudo')" style="background:#6c757d;color:#fff;">Cancelar</button>
        <button type="submit" style="background:#dc3545;color:#fff;" id="btnConfirmarLimpeza" disabled>Excluir Tudo</button>
      </div>
    </form>
  </div>
</div>

<script>
  function excluirTransacao(id, descricao){
    document.getElementById('idTransacao').value = id;
    document.getElementById('descricaoTransacao').textContent = descricao;
    document.getElementById('modalExcluirTransacao').style.display = 'block';
  }
  function abrirModalLimparTudo(){
    document.getElementById('modalLimparTudo').style.display = 'block';
  }
  function fecharModal(id){ document.getElementById(id).style.display='none'; }

  // Fechar modal clicando fora
  window.onclick = function(e){
    if(e.target.classList && e.target.classList.contains('modal')){
      e.target.style.display='none';
    }
  };

  // Habilitar botão de exclusão total
  document.getElementById('confirmarLimpeza').addEventListener('change', function(){
    document.getElementById('btnConfirmarLimpeza').disabled = !this.checked;
  });

  // Mostrar/esconder campos extras conforme tipo
  const selectTipo = document.getElementById('tipo_origem');
  const groupEntidade = document.getElementById('group-entidade');

  function handleTipo(){
    groupEntidade.style.display = this.value.includes('RECEITA') ? 'block' : 'none';
  }
  selectTipo.addEventListener('change', handleTipo);
  handleTipo.call(selectTipo);
</script>

</body>
</html>
<?php if (isset($conexao) && $conexao instanceof mysqli) { $conexao->close(); } ?>