<?php
require_once __DIR__ . '/../helpers/AuthHelper.php';
require_once __DIR__ . '/../models/Usuario.php';
require_once __DIR__ . '/../models/Eleitor.php';
require_once __DIR__ . '/../models/Documento.php';

/**
 * Controller do Painel Administrativo
 * Área restrita para administradores gerenciarem usuários e configurações do sistema
 */
class PaineladminController {

    /**
     * Página principal do painel admin
     */
    public function index() {
        // Verifica se está logado
        AuthHelper::requireLogin();
        
        // Verifica se é administrador
        if (!isset($_SESSION['usuario_nivel_acesso']) || $_SESSION['usuario_nivel_acesso'] !== 'Admin') {
            $_SESSION['erro_acesso'] = "Acesso negado. Apenas administradores podem acessar esta área.";
            header('Location: ' . BASE_URL . '/agendas');
            exit();
        }

        // Busca estatísticas gerais do sistema
        $usuarioModel = new Usuario();
        $eleitorModel = new Eleitor();
        $documentoModel = new Documento();

        $dados = [
            'totalUsuarios' => $this->getTotalUsuarios(),
            'totalEleitores' => $this->getTotalEleitores(),
            'totalDocumentos' => $this->getTotalDocumentos(),
            'usuariosRecentes' => $this->getUsuariosRecentes()
        ];

        $pageTitle = "Painel Administrativo";

        require_once __DIR__ . '/../views/partials/header.php';
        require_once __DIR__ . '/../views/paineladmin/index.php';
        require_once __DIR__ . '/../views/partials/footer.php';
    }

    /**
     * Página de gerenciamento de usuários
     */
    public function usuarios() {
        AuthHelper::requireLogin();
        if (!isset($_SESSION['usuario_nivel_acesso']) || $_SESSION['usuario_nivel_acesso'] !== 'Admin') {
            header('Location: ' . BASE_URL . '/agendas');
            exit();
        }

        $usuarioModel = new Usuario();

        // Filtros
        $q = isset($_GET['q']) ? trim($_GET['q']) : '';
        $nivel = isset($_GET['nivel']) ? trim($_GET['nivel']) : '';
        $status = isset($_GET['status']) ? trim($_GET['status']) : '';
        $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
        $limit = 10;
        $offset = ($page - 1) * $limit;
        $sortBy = isset($_GET['sortBy']) ? $_GET['sortBy'] : 'data_criacao';
        $sortDir = isset($_GET['sortDir']) ? $_GET['sortDir'] : 'DESC';

        $usuarios = $usuarioModel->getAllFiltered($q, $nivel, $status, $offset, $limit, $sortBy, $sortDir);
        $total = $usuarioModel->getFilteredCount($q, $nivel, $status);
        $totalPages = max(1, (int)ceil($total / $limit));

        $niveisAcesso = $usuarioModel->getNiveisAcesso();
        $niveisDistinct = $usuarioModel->getDistinctNiveis();

        $pageTitle = "Gerenciar Usuários";

        require_once __DIR__ . '/../views/partials/header.php';
        require_once __DIR__ . '/../views/paineladmin/usuario.php';
        require_once __DIR__ . '/../views/partials/footer.php';
    }

    /**
     * Criar novo usuário
     */
    public function criarUsuario() {
        AuthHelper::requireLogin();
        
        if (!isset($_SESSION['usuario_nivel_acesso']) || $_SESSION['usuario_nivel_acesso'] !== 'Admin') {
            header('Location: ' . BASE_URL . '/agendas');
            exit();
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $nome = trim($_POST['nome']);
            $email = trim($_POST['email']);
            $senha = $_POST['senha'];
            $nivel_acesso = $_POST['nivel_acesso'];

            // Validações básicas
            if (empty($nome) || empty($email) || empty($senha) || empty($nivel_acesso)) {
                $_SESSION['error_message'] = "Todos os campos são obrigatórios.";
                header('Location: ' . BASE_URL . '/paineladmin/criarUsuario');
                exit();
            }

            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $_SESSION['error_message'] = "Email inválido.";
                header('Location: ' . BASE_URL . '/paineladmin/criarUsuario');
                exit();
            }

            $usuarioModel = new Usuario();
            
            if ($usuarioModel->criar($nome, $email, $senha, $nivel_acesso)) {
                $_SESSION['success_message'] = "Usuário criado com sucesso!";
                header('Location: ' . BASE_URL . '/paineladmin/usuarios');
            } else {
                $_SESSION['error_message'] = "Erro ao criar usuário. O email '{$email}' já está em uso ou ocorreu um erro interno.";
                header('Location: ' . BASE_URL . '/paineladmin/criarUsuario');
            }
            exit();
        }

        $pageTitle = "Criar Novo Usuário";

        require_once __DIR__ . '/../views/partials/header.php';
        require_once __DIR__ . '/../views/paineladmin/criar-usuario.php';
        require_once __DIR__ . '/../views/partials/footer.php';
    }

    /**
     * Página de gerenciamento de níveis de acesso
     */
    public function niveisAcesso() {
        AuthHelper::requireLogin();
        if (!isset($_SESSION['usuario_nivel_acesso']) || $_SESSION['usuario_nivel_acesso'] !== 'Admin') {
            header('Location: ' . BASE_URL . '/agendas');
            exit();
        }

        $usuarioModel = new Usuario();
        $niveisAcesso = $usuarioModel->getNiveisAcesso();
        
        // Buscar estatísticas de usuários por nível
        $estatisticas = [];
        foreach ($niveisAcesso as $nivel => $descricao) {
            $estatisticas[$nivel] = $usuarioModel->contarUsuariosPorNivel($nivel);
        }

        $pageTitle = "Gerenciar Níveis de Acesso";

        require_once __DIR__ . '/../views/partials/header.php';
        require_once __DIR__ . '/../views/paineladmin/niveis-acesso.php';
        require_once __DIR__ . '/../views/partials/footer.php';
    }

    /**
     * Atualizar nível de acesso de um usuário
     */
    public function atualizarNivelAcesso() {
        AuthHelper::requireLogin();
        if (!isset($_SESSION['usuario_nivel_acesso']) || $_SESSION['usuario_nivel_acesso'] !== 'Admin') {
            header('Location: ' . BASE_URL . '/agendas');
            exit();
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $usuarioId = $_POST['usuario_id'];
            $novoNivel = $_POST['nivel_acesso'];

            $usuarioModel = new Usuario();
            
            if ($usuarioModel->atualizarNivelAcesso($usuarioId, $novoNivel)) {
                $_SESSION['success_message'] = "Nível de acesso atualizado com sucesso!";
            } else {
                $_SESSION['error_message'] = "Erro ao atualizar nível de acesso.";
            }
        }
        
        header('Location: ' . BASE_URL . '/paineladmin/usuarios');
        exit();
    }

    public function toggleUsuario() {
        AuthHelper::requireLogin();
        if (!isset($_SESSION['usuario_nivel_acesso']) || $_SESSION['usuario_nivel_acesso'] !== 'Admin') {
            header('Location: ' . BASE_URL . '/agendas');
            exit();
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $usuarioId = isset($_POST['usuario_id']) ? intval($_POST['usuario_id']) : 0;
            if ($usuarioId <= 0) {
                $_SESSION['error_message'] = 'ID de usuário inválido.';
                header('Location: ' . BASE_URL . '/paineladmin/usuarios');
                exit();
            }
            if ($usuarioId == $_SESSION['usuario_id']) {
                $_SESSION['error_message'] = 'Você não pode alterar seu próprio status.';
                header('Location: ' . BASE_URL . '/paineladmin/usuarios');
                exit();
            }
            $usuarioModel = new Usuario();
            if ($usuarioModel->toggleStatus($usuarioId)) {
                $_SESSION['success_message'] = 'Status do usuário atualizado com sucesso.';
            } else {
                $_SESSION['error_message'] = 'Erro ao atualizar status do usuário.';
            }
        }

        header('Location: ' . BASE_URL . '/paineladmin/usuarios');
        exit();
    }

    // Métodos auxiliares para buscar dados
    private function getTotalUsuarios() {
        try {
            $pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $stmt = $pdo->query("SELECT COUNT(*) FROM usuarios WHERE ativo = 1");
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            return 0;
        }
    }

    private function getTotalEleitores() {
        try {
            $pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $stmt = $pdo->query("SELECT COUNT(*) FROM eleitores");
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            return 0;
        }
    }

    private function getTotalDocumentos() {
        try {
            $pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $stmt = $pdo->query("SELECT COUNT(*) FROM documentos");
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            return 0;
        }
    }

    private function getUsuariosRecentes() {
        try {
            $pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $stmt = $pdo->query("SELECT nome, email, nivel_acesso, data_criacao FROM usuarios WHERE ativo = 1 ORDER BY data_criacao DESC LIMIT 5");
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }

    /**
     * Formulário de edição de usuário
     */
    public function editarUsuario($id = null) {
        AuthHelper::requireLogin();
        if (!isset($_SESSION['usuario_nivel_acesso']) || $_SESSION['usuario_nivel_acesso'] !== 'Admin') {
            header('Location: ' . BASE_URL . '/agendas');
            exit();
        }

        $id = isset($id) ? intval($id) : 0;
        if ($id <= 0) {
            $_SESSION['error_message'] = 'ID de usuário inválido.';
            header('Location: ' . BASE_URL . '/paineladmin/usuarios');
            exit();
        }

        $usuarioModel = new Usuario();
        $usuario = $usuarioModel->findById($id);
        if (!$usuario) {
            $_SESSION['error_message'] = 'Usuário não encontrado.';
            header('Location: ' . BASE_URL . '/paineladmin/usuarios');
            exit();
        }

        $niveisAcesso = $usuarioModel->getNiveisAcesso();
        $pageTitle = "Editar Usuário";

        require_once __DIR__ . '/../views/partials/header.php';
        require_once __DIR__ . '/../views/paineladmin/editar-usuario.php';
        require_once __DIR__ . '/../views/partials/footer.php';
    }

    /**
     * Atualiza os dados do usuário (nome, email, nível e senha opcional)
     */
    public function atualizarUsuario() {
        AuthHelper::requireLogin();
        if (!isset($_SESSION['usuario_nivel_acesso']) || $_SESSION['usuario_nivel_acesso'] !== 'Admin') {
            header('Location: ' . BASE_URL . '/agendas');
            exit();
        }

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/paineladmin/usuarios');
            exit();
        }

        $id = isset($_POST['usuario_id']) ? intval($_POST['usuario_id']) : 0;
        $nome = isset($_POST['nome']) ? trim($_POST['nome']) : '';
        $email = isset($_POST['email']) ? trim($_POST['email']) : '';
        $nivel = isset($_POST['nivel_acesso']) ? trim($_POST['nivel_acesso']) : '';
        $senha = isset($_POST['senha']) ? $_POST['senha'] : null;

        if ($id <= 0 || empty($nome) || empty($email) || empty($nivel)) {
            $_SESSION['error_message'] = 'Todos os campos obrigatórios devem ser preenchidos.';
            header('Location: ' . BASE_URL . '/paineladmin/editarUsuario/' . $id);
            exit();
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $_SESSION['error_message'] = 'Email inválido.';
            header('Location: ' . BASE_URL . '/paineladmin/editarUsuario/' . $id);
            exit();
        }

        $usuarioModel = new Usuario();
        $ok = $usuarioModel->atualizarUsuario($id, $nome, $email, $nivel, $senha);
        if ($ok) {
            $_SESSION['success_message'] = 'Usuário atualizado com sucesso!';
            header('Location: ' . BASE_URL . '/paineladmin/usuarios');
        } else {
            $_SESSION['error_message'] = 'Erro ao atualizar usuário. O email informado já está em uso ou ocorreu um erro interno.';
            header('Location: ' . BASE_URL . '/paineladmin/editarUsuario/' . $id);
        }
        exit();
    }
}
